require 'div'
require 'div/login'
require 'singleton'
require 'date'
require 'nkf'
require 'fake_login'
require 'koya'

module KoyaCal
  class Event < Koya::KoyaObject
    def initialize(text=nil, estimate=nil, actual=nil)
      self.text = text
      self.estimate = estimate
      self.actual = actual
    end
    koya_attr :text, :estimate, :actual

    def update(hash)
      self.text = hash[:text] if hash.include?(:text)
      self.estimate = hash[:estimate] if hash.include?(:estimate)
      self.actual = hash[:actual] if hash.include?(:actual)
    end

    def ==(other)
      self.class == other.class &&
      self.text == other.text &&
      self.actual == other.actual &&
      self.estimate == other.estimate
    end
  end

  class EventTable < Koya::KoyaObject
    include Koya::KoyaDictMixin

    def store(date, event)
      self[date.to_s] = event
    end

    def delete(date)
      super(date.to_s)
    end

    def fetch(date)
      self[date.to_s]
    end

    def query_month(year, month)
      head = Date.new(year, month)
      tail = (head >> 1) - 1

      ary = []
      query(head, tail) do |date, event|
        ary.push([date, event])
      end
      ary
    end

    def query(head, tail)
      transaction do
        head.step(tail, 1) do |date|
          yield(date, fetch(date))
        end
      end
    end
  end

  class EventDB
    include Singleton

    def initialize
      root = Koya::Store.new('koyacal').root
      root.transaction do
        @table = root['koyacal']
        unless @table
          root['koyacal'] = Koya::KoyaDict.new
          @table = root['koyacal']
        end
      end
    end

    def [](name)
      @table.transaction do
        @table[name] ||= EventTable.new
      end
    end

    def query_month(name, year, month)
      @table.transaction do
        @table[name].query_month(year, month)
      end
    end

    def fetch(name, date)
      @table.transaction do
        @table[name].fetch(date)
      end
    end

    def delete(name, date)
      @table.transaction do
        @table[name].delete(date)
      end
    end

    def update(name, date, hash)
      @table.transaction do
        table = @table[name]
        event = table.fetch(date)
        unless event
          event = Event.new
          table.store(date, event)
        end
        event.update(hash)
      end
    end
  end

  class DivCalSession < Div::TofuSession
    def initialize(bartender, hint=nil)
      super(bartender, hint)
      @login = Login.new
      @base = BaseDiv.new(self)
    end
    attr_reader :login

    def hint
      if @login.login? && !@login.guest?
        @login.user
      else
        super
      end
    end

    def do_GET(context)
      update_div(context)
      context.res_header('content-type', 'text/html; charset=euc-jp')
      context.res_body(@base.to_html(context))
    end

    def kconv(str)
      NKF.nkf('-e', str.to_s)
    end
  end

  class LoginDiv < Div::LoginDiv
    set_erb('login.erb')
  end

  class BaseDiv < Div::Div
    set_erb('base.erb')

    def initialize(session)
      super(session)
      @cal = DivCalDiv.new(session)
      @login = LoginDiv.new(session, session.login, session.hint)
    end
  end

  class DivCalDiv < Div::Div
    set_erb('divcal.erb')

    class BGAttr
      def initialize(colors = ["#eeeeee", "#ffffff"])
        @colors = colors
        @cur = -1
      end

      def succ
        @cur = (@cur+1) % @colors.size
      end

      def to_s
        succ
        %Q+bgcolor="#{@colors[@cur]}"+
      end
    end

    def initialize(session)
      super(session)
      @cal = Front.new
      @div_seq = self.object_id.to_i
      @curr = Date.today
    end

    def query
      @cal.query_month(user, @curr.year, @curr.month)
    end

    def each
      empty_event = Event.new
      query.each do |date, event|
        yield(date, event || empty_event)
      end
    end

    def user
      return nil unless @session.login.login?
      @session.login.user
    end

    def to_args(param)
      date ,= param['date']
      text ,= param['text']
      estimate ,= param['estimate']
      actual ,= param['actual']

      args = {}

      args[:date] = Date.parse(date.to_s) rescue nil
      args[:text] = @session.kconv(text) if text
      if estimate.to_s.size > 0
        args[:estimate] = estimate.to_f rescue nil 
      end
      if actual.to_s.size > 0
        args[:actual] = actual.to_f rescue nil if actual
      end

      args
    end

    def do_delete(context, param)
      args = to_args(param)
      @cal.delete(user, args[:date])
    rescue
    end

    def do_update(context, param)
      args = to_args(param)
      @cal.update(user, @curr, args)
    rescue
    end

    def do_detail(context, param)
      args = to_args(param)
      @curr = args[:date] if args[:date]
    end
  end
end

if __FILE__ == $0
  tofu = Tofu::Bartender.new(DivCal::DivCalSession)
  DRb.start_service('druby://localhost:12345', tofu)
  DRb.thread.join
end
