require 'test/unit'
require 'korho'
require 'stringio'

class TestWorkspace < Test::Unit::TestCase
  def setup
    File.unlink('test.db') rescue nil
    @world = Korho::World.new('test.db')
  end

  def teardown
  end
  
  def test_00
    assert_equal(0, @world.revisions.size)
    @world.setup_workspace
    assert_equal(0, @world.revisions.size)
    w = @world['workspace']
    w.src = 'value'
    assert_equal('value', w.src)
    assert_equal(1, @world.revisions.size)
    w.src = "value\n  vogue"
    assert_equal("value\nvogue", w.src)
    assert_equal(2, @world.revisions.size)
    tag = @world.revisions[-1]
    @world.look_back(tag) do
      assert_equal('value', w.src)
    end
  end

  def test_01
    s = @world['someday']
    s.add(%w(1 2 3 4 5).join("\n"))
    s.add(%w(6 7 8 9 10).join("\n"))
    assert_equal("1 2 3 4 5 6 7 8 9 10", s.collect {|x| x.to_s}.join(" "))
    
    ten = s.find {|x| x.to_s == '10'}

    ten.move_right
    assert_equal("1 2 3 4 5 6 7 8 9 10", s.collect {|x| x.to_s}.join(" "))

    ten.move_left
    assert_equal("1 2 3 4 5 6 7 8 10 9", s.collect {|x| x.to_s}.join(" "))

    s.transaction do
      ten.move_left
      s.delete(ten)
      assert_equal("1 2 3 4 5 6 7 8 9", s.collect {|x| x.to_s}.join(" "))

      s.unshift(ten)
      assert_equal("10 1 2 3 4 5 6 7 8 9", s.collect {|x| x.to_s}.join(" "))

      8.times { ten.move_right }
      assert_equal("1 2 3 4 5 6 7 8 10 9", s.collect {|x| x.to_s}.join(" "))
    end
  end

  def test_02
    data = <<EOS
= workspace
o Hello, World.
x printf
o include
= trash
= someday
o Hello, Again.
= reference
= project
= delegate
= defer
o Carib
EOS
    @world.import(StringIO.new(data))
    out = ""
    @world.export(out)
    assert_equal(data, out)
  end
end

class TestNode < Test::Unit::TestCase
  def setup
    File.unlink('test.db') rescue nil
    @world = Korho::World.new('test.db')
  end

  def teardown
  end
  
  def test_bg
    @world.setup_workspace
    @world.root.transaction do
      @world.root['node'] = Korho::Node.new('Hello, World')
    end
    node = @world.root['node']
    assert_equal('Hello, World', node.to_s)

    node.color_seed = 0b111111111111111
    assert_equal('#d8d8d8', node.bgcolor)

    node.color_seed = 0
    assert_equal('#f7f7f7', node.bgcolor)
    assert_equal('#ea0300', node.bgcolor(false))
  end
end
