require 'div'
require 'nkf'
require 'korho'
require 'singleton'
require 'korho_style'

require 'hatena/api/auth'
require 'korho_hatena_key'

module KorhoDiv
  class TheWorld < Korho::World
    include Singleton
    def initialize
      super('korho.db')
    end
  end

  class KorhoSession < Div::TofuSession
    def initialize(bartender, hint=nil)
      super(bartender, hint)
      @base = BaseDiv.new(self)
      @hatena_auth = hatena_auth
      @login = nil
    end
    attr_reader :login

    def do_GET(context)
      hatena_cert(context.req_params)

      update_div(context)
      context.res_header('content-type', 'text/html; charset=euc-jp')
      context.res_body(@base.to_div(context))
    end

    def login_uri
      @hatena_auth.uri_to_login
    end

    def hatena_cert(param)
      cert ,= param['cert']
      if cert
        begin
          @login = @hatena_auth.login(cert)
        rescue
          p $!
        end
      end
    end

    def hatena_auth
      Hatena::API::Auth.new(:api_key => API_KEY, :secret => SECRET)
    end
    
    def kconv(str)
      NKF.nkf('-e', str.to_s)
    end
  end

  class KorhoUI
    def initialize
      @korho = TheWorld.instance
      @drawer = @korho['workspace']
      @css = nil
      @curr = nil
      @time_travel = nil
    end
    attr_reader :korho, :drawer, :time_travel
    attr_writer :curr
    attr_accessor :css

    def transaction_for_read
      value = nil
      @korho.root.transaction do |h|
        @korho.koya.revert_to(@time_travel.curr) if @time_travel
        value = yield(h)
        raise Korho::Abort
      end
    rescue Korho::Abort
    ensure
      return value
    end

    def curr
      if @time_travel
        nil
      else
        @curr
      end
    end

    def drawer=(v)
      @drawer = v
      @curr = nil
    end

    def key_to_drawer(key)
      @korho[key]
    end

    def key_to_node(key)
      @drawer.each do |x|
        return x if x.koya_id == key
      end
      nil
    end

    def move_to_drawer(dest)
      return unless @curr
      @curr.transaction do
        @drawer.delete(@curr)
        dest.push(@curr)
      end
      @drawer = dest
    end

    def begin_time_travel
      @time_travel = Korho::TimeTravel.new(@korho)
    end

    def move_time_travel(sec)
      return unless @time_travel
      @time_travel.move_grid(sec)
    end

    def end_time_travel
      @time_travel = nil
    end
  end

  class BaseDiv < Div::Div
    set_erb('base.erb')

    def initialize(session)
      super(session)
      @korho = KorhoUI.new
      @main = ListDiv.new(session, @korho)
      @add = AddDiv.new(session, @korho)
      @drawer = DrawerDiv.new(session, @korho)
      @style = TDiaryStyleDiv.new(session, @korho)
      @travel = TravelDiv.new(session, @korho)
    end

    def to_div(context)
      @korho.transaction_for_read do
        to_html(context)
      end
    end
  end

  class DrawerDiv < Div::Div
    set_erb('korho_drawer.erb')

    def initialize(session, korho)
      super(session)
      @korho = korho
    end

    def to_args(param)
      drawer ,= param['drawer']
      args = {}
      args[:drawer] = drawer if drawer.to_s.size > 0
      args
    end

    def set_drawer(found)
      @korho.drawer = found if found
    end

    def time_travel
      @korho.begin_time_travel
    end

    def do_drawer(context, param)
      args = to_args(param)
      return unless args[:drawer]
      found = @korho.key_to_drawer(args[:drawer])
      if found
        set_drawer(found)
      elsif args[:drawer] == 'time_travel'
        time_travel
      end
    rescue
      p $!
    end
  end

  class MoveToDrawerDiv < DrawerDiv
    def set_drawer(found)
      @korho.move_to_drawer(found) if found
    end
  end

  class AddDiv < Div::Div
    set_erb('korho_add.erb')

    def initialize(session, korho)
      super(session)
      @korho = korho
    end

    def to_args(param)
      src ,= param['src']
      args = {}
      args[:src] = src if src.to_s.size > 0
      args
    end

    def do_add(context, param)
      args = to_args(param)
      return unless args[:src]
      @korho.drawer.add(args[:src])
    rescue
      p $!
    end
  end
  
  class ListDiv < Div::Div
    set_erb('korho_list.erb')

    def initialize(session, korho)
      super(session)
      @korho = korho
      @drawer_div = MoveToDrawerDiv.new(session, korho)
    end

    def curr
      @korho.curr
    end

    def drawer
      @korho.drawer
    end

    def to_args(param)
      key ,= param['key']
      date ,= param['date']
      text ,= param['text']
      estimate ,= param['estimate']
      actual ,= param['actual']
      drawer ,= param['drawer']
      done ,= param['done']

      args = {}

      args[:key] = key.to_s if key
      args[:date] = Date.parse(date.to_s) rescue nil
      args[:text] = @session.kconv(text) if text
      if estimate.to_s.size > 0
        args[:estimate] = estimate.to_f rescue nil 
      end
      if actual.to_s.size > 0
        args[:actual] = actual.to_f rescue nil if actual
      end
      args[:drawer] = drawer if drawer.to_s.size > 0
      args[:done] = (done.to_s == 'done')

      args
    end

    def do_detail(context, param)
      return if @korho.time_travel
      args = to_args(param)
      return unless args[:key]
      @korho.curr = @korho.key_to_node(args[:key])
    rescue
      p $!
    end

    def do_up(context, param)
      return if @korho.time_travel
      args = to_args(param)
      return unless args[:key]
      node = @korho.key_to_node(args[:key])
      node.move_left if node
    rescue
      p $!
    end

    def do_down(context, param)
      return if @korho.time_travel
      args = to_args(param)
      return unless args[:key]
      node = @korho.key_to_node(args[:key])
      node.move_right if node
    rescue
      p $!
    end

    def do_color(context, param)
      return if @korho.time_travel
      @korho.curr.setup_color if @korho.curr
    rescue
      p $!
    end

    def do_done(context, param)
      return if @korho.time_travel
      args = to_args(param)
      return unless args[:key]
      node = @korho.key_to_node(args[:key])
      return unless node
      if args[:done]
        node.set_done
      else
        node.reset_done
      end
    rescue
      p $!
    end

    def do_update(context, param)
      return if @korho.time_travel
      args = to_args(param)
      return unless args[:key]
      node = @korho.key_to_node(args[:key])
      return unless node
      node.str = args[:text]
    rescue
      p $!
    end

    def do_drawer(context, param)
      return if @korho.time_travel
      args = to_args(param)
      return unless args[:drawer]
      found = @korho.key_to_drawer(args[:drawer])
      @korho.move_to_drawer(found) if found
    rescue
      p $!
    end

    def table_bgcolor
      if curr
        "bgcolor='#{curr.bgcolor(false)}'"
      else
        ''
      end
    end

    def make_url(method_name, add_param, context)
      param = make_param(method_name, add_param)
      ary = param.collect do |k, v|
	"#{u(k)}=#{u(v)}"
      end
      %Q!#{action(context)}?#{ary.join(';')}!
    end

    def each
      @korho.drawer.each do |x|
        yield(x)
      end
    end
  end

  class TravelDiv < Div::Div
    set_erb('korho_travel.erb')

    def initialize(session, korho)
      super(session)
      @korho = korho
    end

    def to_args(param)
      delta ,= param['delta']

      args = {}

      args[:delta] = delta.to_f if delta
      args
    end

    def do_travel(context, param)
      args = to_args(param)
      return unless args[:delta]
      @korho.move_time_travel(args[:delta])
    end

    def do_quit(context, param)
      @korho.end_time_travel
    end
  end
end

if __FILE__ == $0
  tofu = Tofu::Bartender.new(KorhoDiv::KorhoSession)
  DRb.start_service('druby://localhost:12345', tofu)
  # DRb.thread.join
  while true
    gets
    KorhoDiv::BaseDiv.reload_erb
    KorhoDiv::DrawerDiv.reload_erb
    KorhoDiv::AddDiv.reload_erb
    KorhoDiv::ListDiv.reload_erb
    KorhoDiv::TDiaryStyleDiv.reload_erb
    KorhoDiv::TravelDiv.reload_erb
  end
end
