require 'koya'

module Korho
  class Abort < RuntimeError; end

  class World
    def initialize(fname)
      @koya = Koya::Store.new(fname)
      @root = @koya.root
      @drawer = nil
      setup_workspace
    end
    attr_reader :root, :drawer, :koya

    def [](name)
      @drawer[name]
    end

    def revisions
      @koya.revisions
    end

    def look_back(tag)
      @koya.transaction do 
        @koya.revert_to(tag)
        yield(@root)
        raise Abort
      end
    rescue Abort
      nil
    end

    def setup_workspace
      @root.transaction do
        @drawer = @root['drawer']
        return if @drawer
        @root['drawer'] = @drawer = Koya::Dict.new
        Drawer.drawer_list.each do |name|
          @drawer[name] = Drawer.new(name)
        end
        @koya.gc
      end
    end

    def export(out)
      @root.transaction do
        Drawer.drawer_list.each do |name|
          out << "= #{name}\n"
          @drawer[name].export(out)
        end
      end
    end

    def import(stream)
      @root.transaction do
        drawer = nil
        while line = stream.gets
          if /^(.) (.+)$/ =~ line
            case $1
            when '='
              drawer = @drawer[$2]
            when 'o'
              drawer.push(Node.new($2))
            when 'x'
              node = Node.new($2)
              drawer.push(node)
              node.set_done
            end
          end
        end
      end
    end
  end

  class Node < Koya::KoyaCell
    def initialize(text)
      self.str = text
      self.done = nil
      self.drawer = nil
      self.created = Time.now
      setup_color
    end
    koya_attr :str, :done, :color_seed, :drawer, :created

    def koya_id
      @_koya_.rowid
    end
    
    def to_s
      self.str.to_s
    end

    def set_done
      self.done = Time.now
    end

    def reset_done
      transaction do
        return unless self.done
        self.done = nil
      end
    end

    def setup_color
      self.color_seed = rand(2 ** 15 + 1)
    end
    
    def bgcolor(light = true)
      k = self.color_seed
      base = 247
      r = base - (k & 0b11111)
      g = base - ((k >> 5) & 0b11111)
      b = base - ((k >> 10) & 0b11111)
      unless light
        r, g, b = dark(r, g, b)
      end
      sprintf("#%02x%02x%02x", r, g, b)
    end

    def dark(r, g, b)
      h, s, v = to_HSV(r, g, b)
      s = [s + 0.05, 1.0].min
      v = [v - 0.05, 0].max
      from_HSV(h, s, v)
    end

    private
    def to_HSV(r, g, b)
      r = r / 255.0
      g = g / 255.0
      b = b / 255.0

      max = [r, g, b].max.to_f
      min = [r, g, b].min.to_f

      v = max
      return [v, v, v] if max == min
      
      s = (max - min) / max
      cr = (max - r) / (max - min)
      cg = (max - g) / (max - min)
      cb = (max - b) / (max - min)

      if (r.to_f == max)
        h = cb - cg
      elsif (g.to_f == max)
        h = 2 + cr - cb
      else
        h = 4 + cg - cr
      end
      
      h = 60 * h
      h = h + 360 if h < 0

      [h, s, v]
    end

    def from_HSV(h, s, v)
      r, g, b = from_HSV_f(h, s, v)
      [(255 * r).floor, (255 * g).floor, (255 * b).floor]
    end

    def from_HSV_f(h, s, v)
      return [v, v, v] if s == 0

      i = (h / 60.0).floor
      f = ((h * 6).to_f % 360) / 360.0
      m = v * (1 - s)
      n = v * (1 - s * f)
      k = v * (1 - s * (1 - f))

      case i
      when 0
        [v, k, m]
      when 1
        [n, v, m]
      when 2
        [m, v, k]
      when 3
        [m, n, v]
      when 4
        [k, m, v]
      else
        [v, m, n]
      end
    end
  end

  class Drawer < Koya::KoyaCellList
    def self.drawer_list
      %w(workspace trash someday reference project delegate defer)
    end
    
    def initialize(name)
      self.name = name
    end
    koya_attr :name

    def length
      inject(0) {|s, x| s + 1}
    end

    def export(out)
      each do |x|
        if x.done
          out << "x"
        else
          out << "o"
        end
        out << " #{x.to_s}\n"
      end
    end

    def src
      collect {|x| x.to_s}.join("\n")
    end

    def src=(str)
      transaction do
        while shift
          # nop
        end
        add(str)
      end
    end

    def add(str)
      transaction do
        str.each do |line|
          line = line.strip
          next if line.size == 0
          push(Node.new(line))
        end
      end
    end
  end
  
  class TimeTravel
    def initialize(world)
      set_revisions(world.revisions)
      @curr = @now
    end
    attr_reader :revisions, :curr

    def set_revisions(ary)
      @now = Time.now
      @revisions = [@now] + ary
      @origin = @revisions[-1]
    end

    def position
      (@now - @curr).to_f / (@now - @origin).to_f
    end

    def move(delta)
      @curr += delta
      @curr = [@curr, @now].min
      @curr = [@curr, @origin].max
    end

    def move_grid(delta)
      move(delta)
      if delta > 0
        @revisions.reverse_each do |x|
          if @curr <= x
            @curr = x
            break
          end
        end
      else
        @revisions.each do |x|
          if @curr >= x
            @curr = x
            break
          end
        end
      end
    end
  end
end

