require 'test/unit'
require 'abload/traverse'
require 'abload/marshal_format'
require 'abload/rd_format'

class WriterTest < Test::Unit::TestCase

  module Mod
    @mod = 1
    module A
      module B
	module C
	  class D
	  end
	end
      end
    end
  end

  class Foo
    def initialize(str)
      @hello = str
    end
    attr_reader :hello
    
    def ==(it)
      self.class == it.class && it.hello == @hello
    end
  end

  class Bar < Foo
    def initialize(str)
      super
      @bod = Time.now
    end
    attr_reader :bod

    def ==(it)
      super(it) && it.bod == @bod
    end
  end

  class Baz
    attr_accessor :baz
    def succ; end
    def <=>( rhs ); object_id <=> rhs.object_id; end
  end

  class FooDump < Foo
    def initialize(str)
      super(str)
      @not_dump = self.object_id
    end
    attr_reader :not_dump

    def __dump(dumper)
      dumper.put_ivar('@hello', @hello)
    end
  end

  class FooDump2 < FooDump
    def __dump(dumper)
      dumper.put_class(Foo)
      dumper.put_ivar('@hello', @hello)
    end
  end

  class DRbRef
    def initialize(uri, ref)
      @uri = uri
      @ref = ref
    end
    attr_reader :uri, :ref
    
    def ==(it)
      self.class == it.class && @uri == it.uri && @ref == it.ref
    end

    def __dump(dumper)
      dumper.put_ivar('@ref', @ref)
    end
  end

  class DRbRefLoad
    def initialize(uri, ref)
      @uri = uri
      @ref = ref
    end
    attr_reader :uri, :ref
    
    def ==(it)
      self.class == it.class && @uri == it.uri && @ref == it.ref
    end

    def __dump(dumper)
      dumper.put_ivar('@ref', @ref)
    end
    
    def self.__load(loader, ref)
      ObjectSpace._id2ref(loader.load(ref.ivars['@ref'.intern]))
    end
  end

  class FooDump3 < Foo
    def __dump(dumper)
      dumper.put_class(DRbRef)
      dumper.put_ivar('@uri', 'druby://foo:12345')
      dumper.put_ivar('@ref', self.__id__)
    end
  end
  
  class FooError < RuntimeError
    def initialize(*arg)
      super
      @at = Time.now
    end
    attr_reader :at
    
    def ==(it)
      self.class == it.class && @at == it.at && message == it.message && backtrace == it.backtrace
    end
  end

  class FooArray < Array
    def initialize(n)
      super(n+n)
      @foo = n
    end
    attr_reader :n

    def self.allocate # oops
      new(0)
    end
  end

  
  def dump_load(obj)
    t = ABLoad::Traverser.new
    t.dump(obj)
    m = ABLoad::MarshalFormatter.new
    m.dump(t.stream[0])
    Marshal.load(m.result)
  end

  def dump_load_eq(obj)
    it = dump_load(obj)
    assert_equal(it, obj)
  end

  def test_int
    dump_load_eq(0)
    dump_load_eq(-10)
    dump_load_eq(0x7fffffff)
    dump_load_eq(0x100000000)
    dump_load_eq(0x77ffffff)
  end

  def test_nil_true_false
    dump_load_eq(nil)
    dump_load_eq(true)
    dump_load_eq(false)
    dump_load_eq(0)
  end

  def test_float
    dump_load_eq(-0.101)
    dump_load_eq(0.1)
    dump_load_eq(1.0)
  end

  def test_bignum
    dump_load_eq(111111111111111111)
    dump_load_eq(2000000000000000000000000000000000000000000)
  end

  def test_module_class
    dump_load_eq(Mod)
    dump_load_eq(self.class)
    dump_load_eq(ABLoad)
    dump_load_eq(Mod::A::B::C)
    dump_load_eq(Mod::A::B::C::D)
  end

  def test_symbol
    dump_load_eq(:symbol)
    dump_load_eq('foo-bar'.intern)
  end

  def test_string
    dump_load_eq('')
    dump_load_eq('String')
  end

  def test_regexp
    dump_load_eq(/^foo/m)
    dump_load_eq(/bar$/ix)
  end

  def test_array
    dump_load_eq([1,2,3])
    dump_load_eq([1,true, false, 2,3, nil])
    dump_load_eq([1,true, false, 2,3, nil, ?a])
    dump_load_eq([:bar, 'String', 4.0, 1000000000])
  end

  def test_hash
    dump_load_eq({1=>2, 2=>3})
    dump_load_eq({:foo, 'string', 4.0, 1000000000})
  end

  S = Struct.new("S", :foo, :bar)
  def test_struct
    s = S
    it = s.new(:bar, :foo)
    dump_load_eq(it)

    it.bar = it
    that = dump_load(it)
    assert_equal(that, that.bar)
  end
  
  def test_range
    dump_load_eq(1..2)
    dump_load_eq(1..3)
  end

  def test_time
    dump_load_eq(Time.now)
  end

  def test_obj
    foo = Foo.new('hello')
    dump_load_eq(foo)
    bar = Bar.new('hi')
    dump_load_eq(bar)
  end

  def test_cycle_hash
    o = {}
    o[o] = o
    o2 = dump_load(o)
    assert_equal(o2[o2].object_id, o2.object_id)
  end

=begin
  def test_cycle_range
    r1 = Baz.new
    r2 = Baz.new
    o = r1..r2
    r1.baz = o
    o2 = dump_load(o)
    assert_equal(o.begin.baz.object_id, o.object_id)
    assert_equal(o2.begin.baz.object_id, o2.object_id)
  end
=end

  def test_custom_dump
    f = FooDump.new('hello')
    f2 = dump_load(f)
    assert(f.not_dump != f2.not_dump)
    
    f = FooDump2.new('hello')
    f2 = dump_load(f)
    assert_equal(f2.class, Foo)
    assert_equal(f.hello, f2.hello)

    f = FooDump3.new('hello')
    f2 = dump_load(f)
    d = DRbRef.new('druby://foo:12345', f.__id__)
    assert_equal(f2, d)
  end
  
  def test_exception
    e = FooError.new('test')
    e2 = dump_load(e)
    assert_equal(e, e2)
    
    begin
      raise(FooError, 'hoge')
    rescue FooError
      e = $!
    end

    e2 = dump_load(e)
    assert_equal(e, e2)
  end

  def test_cannot_dump
    # IO
    io = File.open("/dev/null", "r")
    assert_raise(TypeError) { dump_load(io) }
    io.close
    # Binding
    assert_raise(TypeError) { dump_load(binding) }
    # Continuation
    callcc { |cc| assert_raise(TypeError) { dump_load(cc) }}
    # Dir
    dir = Dir.open('.')
    assert_raise(TypeError) { dump_load(dir) }
    dir.close
    # File::Stat
    s = File.stat('/dev/null')
    assert_raise(TypeError) { dump_load(s) }
    # MatchData
    /Foo/ =~ 'FooBar'
    m = $~
    assert_kind_of(MatchData, m)
    assert_raise(TypeError) { dump_load(m) }
    # Method
    assert_raise(TypeError) { dump_load(self.method(:id)) }
    # Proc
    assert_raise(TypeError) { dump_load(Proc.new {|x| x+x}) }
    # Thread
    assert_raise(TypeError) { dump_load(Thread.current) }
    # ThreadGroup
    assert_raise(TypeError) { dump_load(ThreadGroup.new) }
  end
end

class LoaderTest < WriterTest
  def dump_load(obj)
    t = ABLoad::Traverser.new
    t.dump(obj)
    loader = ABLoad::Loader.new
    loader.load(t.stream[0])
  end
  
  def test_custom_loader
    f = Foo.new('hello')
    d = DRbRefLoad.new('druby://foo:12345', f.__id__)
    d2 = dump_load(d)
    assert_equal(f, d2)
  end

  def test_subclass
    f = FooArray.new(2)
    f2 = dump_load(f)
    assert_equal(f, f2)
    assert_equal(f.n, f2.n)
  end
end

class RDTest < LoaderTest
  def dump_load(obj)
    desc_stream = ABLoad.traverse(obj)

    rd_w = ABLoad::RDFormat::Writer.new
    rd_w.write(desc_stream)

    rd_str = rd_w.output.join('')
    desc_stream = nil # discard

    rd_r = ABLoad::RDFormat::Reader.new
    desc_stream = rd_r.read(rd_str)

    ABLoad.load(desc_stream)
  end
end

class GCLoaderTest < LoaderTest
  def dump_load(obj)
    GC.start
    super
  end
end

class GCRDTest < RDTest
  def dump_load(obj)
    GC.start
    super
  end
end

